<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\User;
class AuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('JWT', ['except' => ['login','signup','me']]);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validate  = $request->validate([
         'email' =>'required',
         'password'  => 'required'
        ]);
        $credentials = request(['email', 'password']);

        if (! $token = auth()->attempt($credentials)) {
            return response()->json(['error' => 'Email or Password are Invalid'], 401);
        }

        return $this->respondWithToken($token);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function me()
    {
        return response()->json(auth()->user());
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        auth()->logout();

        return response()->json(['message' => 'Successfully logged out']);
    }


    public function signup(Request $request){
    	$validate   = $request->validate([
         'email' => 'required|unique:users|max:255',
         'name' => 'required',
         'password' => 'required|min:6|confirmed'

    	]);

    	$data  = array();
    	$data['name']  = $request->name;
    	$data['email']  = $request->email;
    	$data['password']  = Hash::make($request->password);

    	User::insert($data);
    	return $this->login($request);

    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        return $this->respondWithToken(auth()->refresh());
    }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => auth()->factory()->getTTL() * 60,
            'name' =>auth()->user()->name,
            'user_id' =>auth()->user()->id,
            'email' =>auth()->user()->email,
        ]);
    }
}
